# Beta(a_ell,b_ell) on (0,1) or (-1,1) for partial correlation

# Goal is to have constant E(R_{l,l+1}) and E(R^2_{l,l+1}) by C-vine tree
# match E(R_{23}), E(R_{34}) to E(R_{12})
# match E(R^2_{23}), E(R^2_{34}) to E(R^2_{12})
# match E(R_{23}) to E(R_{12})
# etc up to dimension k

# E(R_{23} = mu1^2 + mu2 + gam1^2
# E(R^2_{23} = nu2 *(1-nu1)^2 + nu1^2 + 2 *mu2 *eta1^2
# E(R_{34} = mu1^2 + mu2^2 *gam1^2 + mu3 *gam1^2 *gam2^2
# E(R^2_{34} = nu3 *(1-nu2)^2 *(1-nu1)^2 + nu2^2 *(1-nu1)^2 + nu1^2 
#   + 2 *mu3 *eta2^2 *(1-nu1)^2 + 2 *mu3 *gam2^2 *eta1^2
#   + 2 *mu2^2 *eta1^2
# etc

#======================================================================

#' Recursion for a[ell], b[ell] of Beta random variables in (0,1) for partial
#' correlations in row/tree ell starting from positive Beta(a1,b1) distribution
#' for correlations in row 1 of C-vine, so that all correlations have mean
#' and SD of a Beta(a1,b1) random variable in (0,1) 
#'
#' @description
#' Get (a[ell],b[ell]) for Beta parameters of random partial correlations in 
#' tree ell=2,...,K,
#' starting from Beta(a1,b1) in (0,1) in row 1 of C-vine,
#' for the goal to have constant E(R_[ell,ell+1]) and  
#' E(R^2_[ell,ell+1]) by C-vine tree
#'
#' @param a1 Beta parameter for correlations in C-vine tree 1
#' @param b1 Beta parameter for correlations in C-vine tree 1
#' @param K integer >=2, simulate random correlation matrices to dimension K
#' K=2 get Beta parameter for R_[23;1],
#' K=3 get Beta parameters for R_[23;1], R_[34;12],
#' K general: get Beta parameters for R_[23;1],...,R_[K,K+1;1...K-1]
#' @param iprint TRUE for intermediate prints at tree levels
#'
#' @return Kx2 matrix with Beta parameters avec,bvec; 
#'   feasibility to row i, where i<=K; 
#'   avec[(i+1):K]=0 and bvec[(i+1):K]=0 if not feasible
#'
#' @examples
#' ab = get2betapars_pos(a1=2,b1=2, 3, iprint=TRUE); print(ab)
#' ab = get2betapars_pos(a1=1,b1=1, 3, iprint=TRUE); print(ab)
#' ab = get2betapars_pos(a1=1,b1=0.5, 3, iprint=TRUE) # fails for R_{34}
#' ab = get2betapars_pos(a1=1,b1=2, 3, iprint=TRUE)
#' ab = get2betapars_pos(a1=0.5,b1=0.5, 3, iprint=TRUE) # fails for R_{34}
#' ab = get2betapars_pos(a1=5,b1=5, 6, iprint=TRUE)
#' ab = get2betapars_pos(a1=5,b1=5, 10, iprint=TRUE)
#' ab = get2betapars_pos(a1=5,b1=5, 14, iprint=TRUE)
#' ab = get2betapars_pos(a1=5,b1=8, 6, iprint=TRUE)
#'
#' @references 
#' Joe and Kurowicka (2026), Random correlation matrices generated via partial correlation C-vines. 
#' Journal of Multivariate Analysis. https://doi.org/10.1016/j.jmva.2025.105519
#'
#' @export
#'
get2betapars_pos = function(a1,b1,K, iprint=FALSE)
{ mvec1 = moments_pos(a1,b1)
  mu = rep(0,K); gam = rep(0,K); nu = rep(0,K); eta = rep(0,K)
  mu2 = rep(0,K); gam2 = rep(0,K); nu2 = rep(0,K); eta2 = rep(0,K);
  nu1sq = rep(0,K)
  a = rep(0,K); b = rep(0,K)
  a[1] = a1; b[1] = b1
  mu[1] = mvec1[3]; gam[1] = mvec1[4]; nu[1] = mvec1[5]; eta[1] = mvec1[6]
  E12 = mvec1[c(3,5)]
  mu2[1] = mu[1]^2; gam2[1] = gam[1]^2; nu2[1] = nu[1]^2; eta2[1] = eta[1]^2
  nu1sq[1] = (1-nu[1])^2
  A = matrix(0,K,K)
  mn = rep(0,K) # mean for different trees
  sq = rep(0,K) # square terms for different trees
  mn[1] = mu[1]; sq[1] = nu[1]
  temgam = 1; temsq = 1 
  A[1,1] = sq[1]
  for(i in 2:K)
  { ggiter = function(ab)
    { ai = ab[1]; bi = ab[2]
      if(ai<=0.001 | bi<=0.001) return(3)
      mveci = moments_pos(ai,bi)
      mu[i] = mveci[3]
      nu[i] = mveci[5]
      # general with A matrix
      mn[i] = mn[i-1]+ temgam*(mu2[i-1]-mu[i-1]+mu[i]*gam2[i-1])
      sq[i] = sq[i-1]+ temsq*(nu2[i-1]-nu[i-1]+nu[i]*nu1sq[i-1])
      A[i,i] = sq[i]
      A[i-1,i] = mu[i]
      if(i==2) { A[1,2] = A[1,2]*2*eta2[1] }
      if(i>=3)
      { temnu=1
        for(j in 1:(i-2))
        { A[j,i] = A[j,i-1] +
            (mu2[i-1]-mu[i-1] + mu[i]*gam2[i-1]) *temgam * (2*eta2[j]*temnu)
          temnu = temnu*nu1sq[j]
        }
        A[i-1,i] = A[i-1,i]*2*eta2[i-1]*temnu
      }
      ER2 = sum(A[,i])
      # objective function
      Enew = c(mn[i],ER2)
      ms = mean((E12-Enew)^2)
      ms
    }
    # checking
    #tem = ggiter(c(a1,b1))
    #outsolve = nlm(ggiter, c(a[i-1],b[i-1]), print.level=1)
    outsolve = nlm(ggiter, c(a[i-1],b[i-1]), print.level=0)
    ab = outsolve$estimate
    if(min(ab)< 0.0011 | outsolve$code>=3) { message("fails at ", i); i=i-1; break }
    a[i] = ab[1]; b[i] = ab[2] 
    mveci = moments_pos(a[i],b[i])
    mu[i] = mveci[3]; gam[i] = mveci[4]; nu[i] = mveci[5]; eta[i] = mveci[6]
    mu2[i] = mu[i]^2; gam2[i] = gam[i]^2; nu2[i] = nu[i]^2; eta2[i] = eta[i]^2
    nu1sq[i] = (1-nu[i])^2
    mn[i] = mn[i-1]+ temgam*(mu2[i-1]-mu[i-1]+mu[i]*gam2[i-1])
    sq[i] = sq[i-1]+ temsq*(nu2[i-1]-nu[i-1]+nu[i]*nu1sq[i-1])
    A[i,i] = sq[i]
    A[i-1,i] = mu[i]
    if(i==2) { A[1,2] = A[1,2]*2*eta2[1] }
    if(i>=3)
    { temnu=1
      for(j in 1:(i-2))
      { A[j,i] = A[j,i-1] +
          (mu2[i-1]-mu[i-1] + mu[i]*gam2[i-1]) *temgam * (2*eta2[j]*temnu)
        temnu = temnu*nu1sq[j]
      }
      A[i-1,i] = A[i-1,i]*2*eta2[i-1]*temnu
    }
    temgam = temgam* gam2[i-1] 
    temsq = temsq* nu1sq[i-1]
  }
  if(iprint)
  { cat("avec:", a,"\n"); cat("bvec:",b,"\n")
    # check if moments matched
    out = recursion_pos(avec=a[1:i], bvec=b[1:i], iprint=TRUE)
    cat("\n------------------------------------------------------------\n\n")
  }
  cbind(a,b)
}

#======================================================================


#' Recursion for a[ell], b[ell] of Beta random variables in (-1,1) for partial
#' correlations in row/tree ell starting from Beta(a1,b1) distribution in (-1,1)
#' for correlations in row 1 of C-vine, so that all correlations have mean
#' and SD of a Beta(a1,b1) random variable in (-1,1) 
#'
#' @description
#' Get (a[ell],b[ell]) for Beta parameters of random partial correlations in 
#' tree ell=2,...,K,
#' starting from Beta(a1,b1) in (-1,1) in row 1 of C-vine,
#' for the goal to have constant E(R_[ell,ell+1]) and  
#' E(R^2_[ell,ell+1]) by C-vine tree
#'
#' @param a1 Beta parameter for correlations in C-vine tree 1
#' @param b1 Beta parameter for correlations in C-vine tree 1
#' @param K integer >=2, simulate random correlation matrices to dimension K
#' K=2 get Beta parameter for R_[23;1],
#' K=3 get Beta parameters for R_[23;1], R_[34;12],
#' K general: get Beta parameters for R_[23;1],...,R_[K,K+1;1...K-1]
#' @param iprint TRUE for intermediate prints at tree levels
#'
#' @return Kx2 matrix with Beta parameters avec,bvec; 
#'   feasibility to row i, where i<=K; 
#'   avec[(i+1):K]=0 and bvec[(i+1):K]=0 if not feasible
#'
#' @examples
#' ab = get2betapars_mp1(a1=2,b1=2, 3, iprint=TRUE); print(ab)
#' ab = get2betapars_mp1(a1=1,b1=1, 3, iprint=TRUE) # fails at 3
#' ab = get2betapars_mp1(a1=1,b1=0.5, 3, iprint=TRUE) # fails at R_{34}
#' ab = get2betapars_mp1(a1=1,b1=2, 3, iprint=TRUE) # fails at R_{34}
#' ab = get2betapars_mp1(a1=5,b1=5, 6, iprint=TRUE)
#' ab = get2betapars_mp1(a1=5,b1=5, 10, iprint=TRUE)
#' ab = get2betapars_mp1(a1=5,b1=5, 12, iprint=TRUE) # fails at 11
#' ab = get2betapars_mp1(a1=8,b1=5, 6, iprint=TRUE); print(ab)
#' ab = get2betapars_mp1(a1=5,b1=8, 6, iprint=TRUE)  # fails at 4
#' ab = get2betapars_mp1(a1=6,b1=4.8, 6, iprint=TRUE)
#' ab = get2betapars_mp1(a1=8,b1=6.4, 6, iprint=TRUE)
#'
#' @references 
#' Joe and Kurowicka (2026), Random correlation matrices generated via partial correlation C-vines. 
#' Journal of Multivariate Analysis. https://doi.org/10.1016/j.jmva.2025.105519
#'
#' @export
#'
get2betapars_mp1 = function(a1,b1,K, iprint=FALSE)
{ mvec1 = moments_mp1(a1,b1)
  mu = rep(0,K); gam = rep(0,K); nu = rep(0,K); eta = rep(0,K)
  mu2 = rep(0,K); gam2 = rep(0,K); nu2 = rep(0,K); eta2 = rep(0,K)
  nu1sq = rep(0,K)
  a = rep(0,K); b = rep(0,K)
  a[1] = a1; b[1] = b1
  mu[1] = mvec1[3]; gam[1] = mvec1[4]; nu[1] = mvec1[5]; eta[1] = mvec1[6]
  E12 = mvec1[c(3,5)]
  mu2[1] = mu[1]^2; gam2[1] = gam[1]^2; nu2[1] = nu[1]^2; eta2[1] = eta[1]^2
  nu1sq[1] = (1-nu[1])^2

  A = matrix(0,K,K)
  mn = rep(0,K) # mean for different trees
  sq = rep(0,K) # square terms for different trees
  mn[1] = mu[1]
  sq[1] = nu[1]
  temgam = 1 
  temsq = 1 
  A[1,1] = sq[1]
  for(i in 2:K)
  { ggiter = function(ab)
    { ai = ab[1]; bi = ab[2]
      if(ai<=0.001 | bi<=0.001) return(3)
      mveci = moments_mp1(ai,bi)
      mu[i] = mveci[3]
      nu[i] = mveci[5]
      # general with A matrix
      mn[i] = mn[i-1]+ temgam*(mu2[i-1]-mu[i-1]+mu[i]*gam2[i-1])
      sq[i] = sq[i-1]+ temsq*(nu2[i-1]-nu[i-1]+nu[i]*nu1sq[i-1])
      A[i,i] = sq[i]
      A[i-1,i] = mu[i]
      if(i==2) { A[1,2] = A[1,2]*2*eta2[1] }
      if(i>=3)
      { temnu=1
        for(j in 1:(i-2))
        { A[j,i] = A[j,i-1] +
            (mu2[i-1]-mu[i-1] + mu[i]*gam2[i-1]) *temgam * (2*eta2[j]*temnu)
          temnu = temnu*nu1sq[j]
        }
        A[i-1,i] = A[i-1,i]*2*eta2[i-1]*temnu
      }
      ER2 = sum(A[,i])
      # objective function
      Enew = c(mn[i],ER2)
      ms = mean((E12-Enew)^2)
      ms
    }
    # checking
    #tem = ggiter(c(a1,b1))
    #outsolve = nlm(ggiter, c(a[i-1],b[i-1]), print.level=1)
    outsolve = nlm(ggiter, c(a[i-1],b[i-1]), print.level=0)
    ab = outsolve$estimate
    if(min(ab)< 0.0011 | outsolve$code>=3) { message("fails at ", i); i=i-1; break }
    a[i] = ab[1]; b[i] = ab[2] 
    mveci = moments_mp1(a[i],b[i])
    mu[i] = mveci[3]; gam[i] = mveci[4]; nu[i] = mveci[5]; eta[i] = mveci[6]
    mu2[i] = mu[i]^2; gam2[i] = gam[i]^2; nu2[i] = nu[i]^2; eta2[i] = eta[i]^2
    nu1sq[i] = (1-nu[i])^2
    mn[i] = mn[i-1]+ temgam*(mu2[i-1]-mu[i-1]+mu[i]*gam2[i-1])
    sq[i] = sq[i-1]+ temsq*(nu2[i-1]-nu[i-1]+nu[i]*nu1sq[i-1])
    A[i,i] = sq[i]
    A[i-1,i] = mu[i]
    if(i==2) { A[1,2] = A[1,2]*2*eta2[1] }
    if(i>=3)
    { temnu=1
      for(j in 1:(i-2))
      { A[j,i] = A[j,i-1] +
          (mu2[i-1]-mu[i-1] + mu[i]*gam2[i-1]) *temgam * (2*eta2[j]*temnu)
        temnu = temnu*nu1sq[j]
      }
      A[i-1,i] = A[i-1,i]*2*eta2[i-1]*temnu
    }
    temgam = temgam* gam2[i-1] 
    temsq = temsq* nu1sq[i-1]
  }
  if(iprint)
  { cat("avec:", a,"\n"); cat("bvec:",b,"\n")
    # check if moments matched
    out = recursion_mp1(avec=a[1:i], bvec=b[1:i], iprint=TRUE)
    cat("\n------------------------------------------------------------\n\n")
  }
  cbind(a,b)
}

