
#' With a[ell]=a1, recursion for second parameter b[ell] of Beta random variables 
#' in (0,1) for partial correlations in row/tree ell starting from positive
#' Beta(a1,b) distribution for correlations in row 1 of C-vine,
#' so that all correlations have mean a1/(a1+b)
#'
#' @description 
#' Get mu[ell] and b[ell], with a[ell]=a1, for Beta parameters of random 
#' partial correlations in row/tree ell=2,...,K,
#' starting from Beta(a1,b) in (0,1) in row 1,
#' for the goal that all correlations have mean a1/(a1+b)
#'
#' @param a1 first parameter Beta(a1,b) on (0,1) for correlation in tree/row 1
#' @param b second parameter Beta(a1,b) on (0,1) for correlation in tree/row 1
#' @param K desired level: simulate random correlation matrices to dimension K
#' @param iprint TRUE for intermediate prints at tree levels
#'
#' @return (muvec,bvec) as 2-column matrix:   
#' muvec is vector of mean parameters for random partial correlations in trees 2 to ell, ell<K if K is not feasible; 
#' bvec is vector of second Beta parameter if considering Beta(a1,bvec[ell]) 
#'    for positive partial correlation in tree ell. 
#' For the method of allowing some negative partial correlations, 
#'  use muvec and not bvec.
#'
#' @examples
#' tem1 = getbetapars_pos(a1=1,b=1,K=6,iprint=TRUE)
#' tem2 = getbetapars_pos(a1=1,b=0.5,K=6,iprint=TRUE)
#' tem3 = getbetapars_pos(a1=1,b=0.36,K=6,iprint=TRUE) # increasing bvec
#' tem4 = getbetapars_pos(a1=1,b=0.35,K=6,iprint=TRUE)  # bvec decreasing to 0
#' tem4b = getbetapars_pos(a1=1,b=0.35,K=7,iprint=TRUE)  # message mu_new>=1
#' tem5 = getbetapars_pos(a1=1,b=0.35,K=12,iprint=TRUE) # stops at 7
#' tem6 = getbetapars_pos(a1=2,b=0.32,K=12,iprint=TRUE) # stops at 4
#'
#' @references 
#' Joe and Kurowicka (2026), Random correlation matrices generated via partial correlation C-vines. 
#' Journal of Multivariate Analysis. https://doi.org/10.1016/j.jmva.2025.105519
#' 
#' @export
#'
getbetapars_pos = function(a1,b,K,iprint=FALSE)
{ bvec = rep(0,K) 
  gvec = rep(0,K)
  ER = rep(0,K)
  muvec = rep(0,K)
  out = moments_pos(a1,b)  
  bvec[1] = b
  muvec[1] = out[3]
  gvec[1] = out[4]
  ER[1] = out[3]
  for(ell in 2:K)
  { muprev = muvec[ell-1]
    munew = muprev*(1-muprev)/gvec[ell-1]^2
    muvec[ell] = munew
    #if(munew>=1) { message("mu_new>=1"); return(cbind(muvec,bvec)) }
    if(munew>=1) 
    { message("mu_new>=1"); 
      muvec0 = muvec[1:(ell-1)]; bvec0 = bvec[1:(ell-1)]
      return(cbind(muvec0,bvec0)) 
    }
    bnew = a1*(1-munew)/munew
    if(ell==2) 
    { ER[ell] = ER[ell-1] + muprev^2-muprev+munew*gvec[ell-1]^2 }
    else 
    { ER[ell] = ER[ell-1] + (muprev^2-muprev+munew*gvec[ell-1]^2)*prod(gvec[1:(ell-2)]) }
    # next mu
    out = moments_pos(a1,bnew)  
    gvec[ell] = out[4]
    if(iprint) cat(ell,munew,bnew,ER[ell],"\n")
    bvec[ell] = bnew
  }
  if(iprint) cat("\n")
  cbind(muvec,bvec)
}


#' With a[ell]=a1, recursion for second parameter b[ell] of Beta random variables 
#' in (-1,1) for partial correlations in row/tree ell starting from 
#' Beta(a1,b) distribution in (-1,1) for correlations in row 1 of C-vine,
#' so that all correlations have mean 2*a1/(a1+b)-1
#'
#' @description 
#' Get mu[ell] and b[ell], with a[ell]=a1, for Beta parameters of random 
#' partial correlations in row/tree ell=2,...,K,
#' starting from Beta(a1,b) in (-1,1) in row 1,
#' for the goal that all correlations have mean 2*a1/(a1+b)-1
#'
#' @param a1 first parameter Beta(a1,b) on (-1,1) for correlation in tree/row 1
#' @param b second parameter Beta(a1,b) on (-1,1) for correlation in tree/row 1
#' @param K desired level: simulate random correlation matrices to dimension K
#' @param iprint TRUE for intermediate prints at tree levels
#'
#' @return (muvec,bvec) as 2-column matrix: 
#' muvec is vector of mean parameters for random partial correlations in trees 2 to ell, ell<K if K is not feasible; 
#' bvec is vector of second Beta parameter if considering Beta(a1,bvec[ell]) 
#'    for partial correlation in tree ell
#'
#' @examples
#' tem1 = getbetapars_mp1(a1=3,b=2,K=8,iprint=TRUE)
#' tem2 = getbetapars_mp1(a1=4,b=2,K=8,iprint=TRUE)
#' tem3 = getbetapars_mp1(a1=6,b=2,K=8,iprint=TRUE)
#'
#' @references 
#' Joe and Kurowicka (2026), Random correlation matrices generated via partial correlation C-vines. 
#' Journal of Multivariate Analysis. https://doi.org/10.1016/j.jmva.2025.105519
#' 
#' @export
#'
getbetapars_mp1 = function(a1,b,K,iprint=FALSE)
{ bvec = rep(0,K) 
  gvec = rep(0,K)
  ER = rep(0,K)
  muvec = rep(0,K)
  out = moments_mp1(a1,b)  
  bvec[1] = b
  muvec[1] = out[3]
  gvec[1] = out[4]
  ER[1] = out[3]
  for(ell in 2:K)
  { muprev = muvec[ell-1]
    munew = muprev*(1-muprev)/gvec[ell-1]^2
    muvec[ell] = munew
    if(munew>=1 | munew<= -1) 
    { message("mu_new out of range"); 
      #return(cbind(muvec,bvec)) 
      muvec0 = muvec[1:(ell-1)]; bvec0 = bvec[1:(ell-1)]
      return(cbind(muvec0,bvec0)) 
    }
    bnew = a1*(1-munew)/(munew+1)
    if(ell==2) 
    { ER[ell] = ER[ell-1] + muprev^2-muprev+munew*gvec[ell-1]^2 }
    else 
    { ER[ell] = ER[ell-1] + (muprev^2-muprev+munew*gvec[ell-1]^2)*prod(gvec[1:(ell-2)]) }
    # next mu
    out = moments_mp1(a1,bnew)  
    gvec[ell] = out[4]
    if(iprint) cat(ell,munew,bnew,ER[ell],"\n")
    bvec[ell] = bnew
  }
  if(iprint) cat("\n")
  cbind(muvec,bvec)
}

