# Functions for positive correlation (_pos)
#  and correlations in (-1,1) minus1/plus1 (_mp1)
# Assume partial correlation C-vine 

#======================================================================

# Functions for Beta(a,b) on (0,1)

#' Integrand for E[sqrt(1-B^2)], B~Beta(a,b) on (0,1)
#'
#' @description 
#' Integrand for E[sqrt(1-B^2)], B~Beta(a,b) on (0,1)
#'
#' @param x vector with values in (0,1)
#' @param a Beta distribution first parameter 
#' @param b Beta distribution second parameter 
#'
#' @return sqrt(1-x^2)*pdf
#'
gam_pos = function(x,a,b)
{ pdf = dbeta(x,a,b)
  sqrt(1-x^2)*pdf
}

#' Integrand for E[B*sqrt(1-B^2)], B~Beta(a,b) on (0,1)
#'
#' @description 
#' Integrand for E[B*sqrt(1-B^2)], B~Beta(a,b) on (0,1)
#'
#' @param x vector with values in (0,1)
#' @param a Beta distribution first parameter 
#' @param b Beta distribution second parameter 
#'
#' @return x*sqrt(1-x^2)*pdf
#'
eta_pos = function(x,a,b)
{ pdf = dbeta(x,a,b)
  x*sqrt(1-x^2)*pdf
}

#' For R=B with B~Beta(a,b) on (0,1), get 4 moments
#'
#' @description 
#' For R=B with B~Beta(a,b) on (0,1), get 4 moments
#'
#' @param a Beta distribution first parameter 
#' @param b Beta distribution second parameter 
#'
#' @return 6-vector (a, b, mu, gam, nu, eta) where 
#' mu=E(B), gam=E[sqrt(1-B^2)], nu=E(B^2), eta=E[B*sqrt(1-B^2)]
#'
#' @references 
#' Joe and Kurowicka (2026), Random correlation matrices generated via partial correlation C-vines
#' Journal of Multivariate Analysis
#' 
#' @export
#'
moments_pos = function(a,b)
{ mu = a/(a+b)
  nu = a*(a+1)/(a+b)/(a+b+1)
  gam = integrate(gam_pos,0,1,a=a,b=b)
  eta = integrate(eta_pos,0,1,a=a,b=b)
  # if(iprint) { print(gam); print(eta) }
  gam = gam$value
  eta = eta$value
  c(a,b,mu,gam,nu,eta)
}

#' Recursion for mean and second moment for random correlations in (0,1) by tree of C-vine
#'
#' @description 
#' Recursion for mean and second moment for random correlations in (0,1) by tree
#'
#' @param avec positive vector length d-1>=2; avec[ell] is first Beta parameter for partial correlation in tree ell  
#' @param bvec positive vector length d-1>=2; bvec[ell] is second Beta parameter for partial correlation in tree ell 
#' @param iprint TRUE for intermediate prints
#'
#' @return matrix with columns a, b, ER, ER2, SD, ER_aperm, SD_aperm 
#'     with row length = length(bvec); aperm means after random permutation. 
#'   The output can be used for correlation matrices up to length(bvec)
#'
#' @examples
#' out = recursion_pos(avec=rep(1,11), bvec=seq(1,6,0.5), iprint=TRUE)
#' bb = c(1,1.467401,2.062294,2.748234,3.496715,4.288931,5.112821,5.96039)
#' out = recursion_pos(avec=rep(1,8), bvec=bb, iprint=TRUE)
#'
#' @references 
#' Joe and Kurowicka (2026), Random correlation matrices generated via partial correlation C-vines
#' Journal of Multivariate Analysis
#' 
#' @export
#'
recursion_pos = function(avec, bvec, iprint=FALSE)
{ moment = matrix(0,length(bvec),6)
  for(i in 1:length(bvec))
  { out = moments_pos(avec[i],bvec[i])
    moment[i,] = out
  }
  colnames(moment) = c("a","b","mu","gam","nu","eta")
  mu = moment[,3]; gam = moment[,4]; nu = moment[,5]; eta = moment[,6]
  mu2 = mu^2; gam2 = gam^2; nu2 = nu^2; eta2 = eta^2
  nu1sq = (1-nu)^2
  k = length(mu)
  mn = rep(0,k) # mean for different trees
  sq = rep(0,k) # square terms for different trees
  mn[1] = mu[1]
  sq[1] = nu[1]
  temg = gam2[1]
  temsq = nu1sq[1]
  mn[2] = mu2[1] + mu[2]*temg
  sq[2] = nu2[1] + nu[2]*temsq
  if(k>=3)
  { for(i in 3:k)
    { mn[i] = mn[i-1]+ temg*(mu2[i-1]-mu[i-1]+mu[i]*gam2[i-1])
      sq[i] = sq[i-1]+ temsq*(nu2[i-1]-nu[i-1]+nu[i]*nu1sq[i-1])
      temg = temg* gam2[i-1] 
      temsq = temsq* nu1sq[i-1]
    }
  }
  # Store terms from expected value of cross terms in summands of 
  #  R_{(i)} correlation deduced after tree i.
  # A_{12}, ... A_{1k}  
  # A_{23}, ... A_{2k}  
  # ... A_{k-1,k}
  A=matrix(0,k,k)
  for(i in 1:(k-1)) { A[i,i+1] = mu[i+1] }
  if(k>=3)
  { for(j in 1:(k-2))
    { temg = 1
      for(ell in (j+2):k)
      { A[j,ell] = A[j,ell-1] + (mu2[ell-1]-mu[ell-1] + mu[ell]*gam2[ell-1])*temg
        temg = temg*gam2[ell-1]
      }
    }
  }
  #if(iprint) print(A)
  # Next multiply row by 2*eta2[i] *prod(nu1sq[1:(i-1)])
  temn = 1
  for(i in 1:(k-1))
  { A[i,] = A[i,]*2*eta2[i]*temn
    temn = temn*nu1sq[i]
  }
  #if(iprint) print(A)
  # add diagonal terms from the sq array
  for(i in 1:k) { A[i,i] = sq[i] }
  #if(iprint) print(A)
  # Next second moments by tree come from column sums
  ER2 = apply(A,2,sum)  # ER_{(j)}^2 for trees 1 to k-1
  SDv = sqrt(ER2-mn^2)
  if(iprint) 
  { cat("means, second moments and SDs by tree\n"); 
    print(mn)
    print(ER2) 
    print(SDv)
  }
  # After random permutation, SD of each corr with avec[1:i], bvec[1:i],
  #  i=1,...,length(bvec)
  ub = length(bvec)
  mn_aperm = rep(0,ub)
  SD_aperm = rep(0,ub)
  for(d1 in 1:ub)
  { d = d1+1
    wt = (d1:1)*2/(d*d1)
    mom1 = sum(wt*mn[1:d1])
    mom2 = sum(wt*ER2[1:d1])
    vv = mom2 - mom1^2
    mn_aperm[d1] = mom1
    SD_aperm[d1] = sqrt(vv)
  }
  if(iprint) 
  { cat("means and SDs after random permutation\n") 
    print(mn_aperm)
    print(SD_aperm)
  } 
  mat = cbind(avec,bvec,mn,ER2,SDv,mn_aperm, SD_aperm)
  colnames(mat) = c("a","b","ER", "ER2", "SD","ER_aperm","SD_aperm")
  mat
}

#======================================================================

# Functions for Beta(a,b) on (-1,1)

#' For R=B with B~Beta(a,b) on (-1,1), get 4 moments
#'
#' @description 
#' For R=B with B~Beta(a,b) on (-1,1), get 4 moments
#'
#' @param a Beta distribution first parameter 
#' @param b Beta distribution second parameter 
#'
#' @return 6-vector (a, b, mu, gam, nu, eta) where 
#' mu=E(B), gam=E[sqrt(1-B^2)], nu=E(B^2), eta=E[B*sqrt(1-B^2)]
#'
#' @references 
#' Joe and Kurowicka (2026), Random correlation matrices generated via partial correlation C-vines
#' Journal of Multivariate Analysis
#' 
#' @export
#'
moments_mp1 = function(a,b)
{ mu = 2*a/(a+b)-1
  nu = (4*a*b+(a-b)^2*(a+b+1))/(a+b)^2/(a+b+1)
  den = beta(a,b)
  num1 = beta(a+.5,b+.5)
  #num2 = beta(a+1.5,b+.5)
  egam = 2*num1/den
  #eeta = (4*num2-2*num1)/den
  eeta = egam*(a-b)/(a+b+1)
  #if(iprint) { print(egam); print(eeta) }
  c(a,b,mu,egam,nu,eeta)
}

# Code is same as for _pos with _pos replaced by _mp1


#' Recursion for mean and second moment for random correlations in (-1,1) by tree of C-vine
#'
#' @description 
#' Recursion for mean and second moment for random correlations in (-1,1) by tree
#'
#' @param avec positive vector length d-1>=2; avec[ell] is first Beta parameter for partial correlation in tree ell  
#' @param bvec positive vector length d-1>=2; bvec[ell] is second Beta parameter for partial correlation in tree ell 
#' @param iprint TRUE for intermediate prints
#'
#' @return matrix with columns a, b, ER, ER2, SD, ER_aperm, SD_aperm 
#'     with row length = length(bvec); aperm means after random permutation. 
#'   The output can be used for correlation matrices up to length(bvec)
#'
#' @examples
#' out = recursion_mp1(avec=rep(1,11), bvec=seq(1,6,0.5), iprint=TRUE)
#'
#' @references 
#' Joe and Kurowicka (2026), Random correlation matrices generated via partial correlation C-vines
#' Journal of Multivariate Analysis
#' 
#' @export
#'
recursion_mp1 = function(avec, bvec, iprint=FALSE)
{ moment = matrix(0,length(bvec),6)
  for(i in 1:length(bvec))
  { out = moments_mp1(avec[i],bvec[i])
    moment[i,] = out
  }
  colnames(moment) = c("a","b","mu","gam","nu","eta")
  mu = moment[,3]; gam = moment[,4]; nu = moment[,5]; eta = moment[,6]
  mu2 = mu^2; gam2 = gam^2; nu2 = nu^2; eta2 = eta^2
  nu1sq = (1-nu)^2
  k = length(mu)
  mn = rep(0,k) # mean for different trees
  sq = rep(0,k) # square terms for different trees
  mn[1] = mu[1]
  sq[1] = nu[1]
  temg = gam2[1]
  temsq = nu1sq[1]
  mn[2] = mu2[1] + mu[2]*temg
  sq[2] = nu2[1] + nu[2]*temsq
  if(k>=3)
  { for(i in 3:k)
    { mn[i] = mn[i-1]+ temg*(mu2[i-1]-mu[i-1]+mu[i]*gam2[i-1])
      sq[i] = sq[i-1]+ temsq*(nu2[i-1]-nu[i-1]+nu[i]*nu1sq[i-1])
      temg = temg* gam2[i-1] 
      temsq = temsq* nu1sq[i-1]
    }
  }
  # Store terms from expected value of cross terms in summands of 
  #  R_((i)} correlation deduced after tree i.
  # A_{12}, ... A_{1k}  
  # A_{23}, ... A_{2k}  
  # ... A_{k-1,k}
  A=matrix(0,k,k)
  for(i in 1:(k-1)) { A[i,i+1] = mu[i+1] }
  if(k>=3)
  { for(j in 1:(k-2))
    { temg = 1
      for(ell in (j+2):k)
      { A[j,ell] = A[j,ell-1] + (mu2[ell-1]-mu[ell-1] + mu[ell]*gam2[ell-1])*temg
        temg = temg*gam2[ell-1]
      }
    }
  }
  #if(iprint) print(A)
  # Next multiply row by 2*eta2[i] *prod(nu1sq[1:(i-1)])
  temn = 1
  for(i in 1:(k-1))
  { A[i,] = A[i,]*2*eta2[i]*temn
    temn = temn*nu1sq[i]
  }
  #if(iprint) print(A)
  # add diagonal terms from the sq array
  for(i in 1:k) { A[i,i] = sq[i] }
  #if(iprint) print(A)
  # Next second moments by tree come from column sums
  ER2 = apply(A,2,sum)  # ER_{(j)}^2 for trees 1 to k-1
  SDv = sqrt(ER2-mn^2)
  if(iprint) 
  { cat("means, second moments and SDs by tree\n"); 
    print(mn)
    print(ER2) 
    print(SDv)
  }
  # After random permutation, SD of each corr with avec[1:i], bvec[1:i],
  #  i=1,...,length(bvec)
  ub = length(bvec)
  mn_aperm = rep(0,ub)
  SD_aperm = rep(0,ub)
  for(d1 in 1:ub)
  { d = d1+1
    wt = (d1:1)*2/(d*d1)
    mom1 = sum(wt*mn[1:d1])
    mom2 = sum(wt*ER2[1:d1])
    vv = mom2 - mom1^2
    mn_aperm[d1] = mom1
    SD_aperm[d1] = sqrt(vv)
  }
  if(iprint) 
  { cat("means and SDs after random permutation\n") 
    print(mn_aperm)
    print(SD_aperm)
  } 
  mat = cbind(avec,bvec,mn,ER2,SDv,mn_aperm, SD_aperm)
  colnames(mat) = c("a","b","ER", "ER2", "SD","ER_aperm","SD_aperm")
  mat
}

#======================================================================

# Tests for Beta(a,b) on (0,1)
# out = recursion_pos(avec=rep(1,11), bvec=seq(1,6,0.5), iprint=T)
#cat("\n------------------------------------------------------------\n\n")
#
# bb = c(1,1.467401,2.062294,2.748234,3.496715,4.288931,5.112821,5.96039)
#out = recursion_pos(avec=rep(1,8), bvec=bb, iprint=T)
#cat("\n------------------------------------------------------------\n\n")

# Test for Beta(a,b) on (-1,1)
#out2 = recursion_mp1(avec=rep(1,11), bvec=seq(1,6,0.5), iprint=T)

