#' Control parameters for the EGO algorithm
#'
#' Set the control parameters used by the EGO algorithm. Default values will be used for  unprovided arguments.
#'
#' @param alg A character string specifying the algorithm used to optimize the acquisition function:  "genoud" or "direct".
#' @param rel_tol A numeric value that defines the relative tolerance level to terminate \code{EGO}. The \code{EGO} algorithm stops if the value of the acquisition function is smaller than \code{rel_tol} times the range of \eqn{y(\mathbf{x})} in the initial design for \code{wait_iter} consecutive iterations.  If \code{rel_tol = 0}, \code{EGO} terminates after \code{nstep} iterations. For the definition of \code{nstep}, see the documentation of \code{EGO}.
#' @param wait_iter An integer value of the number of consecutive iterations that the tolerance (\code{rel_tol}) needs to be met before \code{EGO} terminates.
#' @param acq_control A list of two elements:
#'
#' \code{type} is a character string that can be "EI", "AEI", "EQI", or "AKG". When the function to be optimized  is assumed to be deterministic, the expected improvement (EI) is suggested as the acquisition function. The options of "AEI", "EQI", or "AKG" are for the case where the function is evaluated with random noise.
#' See details
#'
#' \code{q} is a numerical value between 0.5 and 1 that specifies the quantile used by AEI and EQI. It is not used by EI or AKG.
#'
#' @param GaSP_control A list of GaSP control parameters that can include (not all required)
#'
#'  \code{cor_family}: the correlation family "PowerExponential" or "Matern".
#'
#'  \code{alpha_min}:  the minimum  the \eqn{\alpha} parameter of the power-exponential correlation.
#'
#'  \code{alpha_max}: the maximum  the \eqn{\alpha} parameter of the power-exponential correlation.
#'
#'  \code{theta_standardized_min}: the minimum of the standardized \eqn{theta} parameter for the power-exponential or the Matern correlation.
#'
#'  \code{theta_standardized_max}: the maximum of the standardized \eqn{theta} parameter for the power-exponential or the Matern correlation.
#'
#'  \code{derivatives_min}: the minimum of the \eqn{\delta} parameter of the  Matern correlation.
#'
#'  \code{derivatives_max}: the maximum of the \eqn{\delta} parameter of the  Matern correlation.
#'
#'  \code{nugget}: a numerical value that proportion of the total variance due to random error is fixed at this value n the deterministic case or bounded below by it in the noisy case. It is recommend to use a small constant for the numerical stability especially for optimizing functions in low dimensions.
#'
#'
#' See the \code{Fit} function in the GaSP package for more details.
#'
#' @param direct_control Only used when \code{alg = "direct"}; a list of control parameters for direct that can include
#'
#' \code{max_eval}: the maximum number of function evaluations.
#'
#' \code{direct_tol}: the tolerance level to terminate the DIRECT algorithm. This tolerance is relative to the range of of \eqn{y(\mathbf{x})} in the initial design.
#'
#' \code{finetune}: a logical value specifying if  L-BFGS-B is performed to refine the solution obtained from DIRECT.
#'
#' See the \code{\link{nloptr}} package for more details on the control parameters.
#'
#' @param optim_control A list of control parameters  passed to \code{optim} if \code{finetune = TRUE} in  \code{direct_control} for the optional finetuning stage after running DIRECT, or passed to genoud for running BFGS.
#'
#' @param genoud_control Only used when \code{alg = "direct"}; a list of control parameters for genoud that can include
#'
#' \code{pop_size}: An integer value of the population size, which  is the number of candidates \code{genoud} uses to solve the optimization problem.
#'
#' \code{max_generations}:  An integer value of the maximum number of generations that \code{genoud} will run when attempting to optimize a function.
#'
#'
#' There are other parameters that the user can set in \code{genoud_control} which usually have less impact on the outcome, e.g.  \code{wait.generations}, \code{BFGSburnin},\code{BFGSmaxit}, \code{P1}, \code{P2}, \code{P3}, \code{P4}, \code{P5}, \code{P6}, \code{P7}, \code{P8}, \code{P9}, \code{P9mix}.
#' For these parameters, see the \code{genoud} function in the \code{rgenoud} package for more details.
#'
#' @param print_level An integer value that controls the level of printing that \code{EGO} does. There are three levels: 0 (minimal printing), 1 (normal with progress at every iteration), 2 (detailed, including warning messages),
#'
#'
#' @return A list of input arguments.
#'
#' @seealso  \code{\link{EGO}}
#'
#'
#' @export
EGO.control <- function(alg = "genoud", rel_tol = 0, wait_iter = 10,
                        acq_control = list(type = "EI"),
                        GaSP_control = list(cor_family = "PowerExponential", alpha_min = 0.0, alpha_max = 1.99, nugget = 1e-09),
                        direct_control = list(max_eval = 3000, direct_tol = 0, finetune = FALSE),
                        genoud_control = list(pop_size = 50, max_generations = 10), optim_control = list(),
                       print_level = 1){
  return(list(alg = alg, rel_tol = rel_tol, wait_iter = wait_iter, acq_control = acq_control,
              GaSP_control = GaSP_control, direct_control = direct_control, genoud_control =  genoud_control, optim_control= optim_control,
              print_level = print_level))
}


#' Efficient Global Optimization (EGO)
#'
#' EGO performs Bayesian optimization using the EGO algorithm.
#' The objective is to find the \eqn{\mathbf{x}} that minimizes \eqn{f(\mathbf{x})}.
#' Two cases are considered: deterministic or noisy.  In the former case,
#' the available function evaluation \eqn{y(\mathbf{x}) = f(\mathbf{x})} is accessible given any query point \eqn{\mathbf{x}}, and in the latter  \eqn{y(\mathbf{x}) = f(\mathbf{x}) + \epsilon} is acceesible  where \eqn{\epsilon} is some random noise.
#'
#' @import lhs nloptr GaSP rgenoud stats graphics grDevices utils
#'
#' @param fun A function to be minimized. It can be deterministic or with added random noise.
#' @param reg_model  A formula object that specifies the regression model passed to the \code{Fit} function in the \code{GaSP} package.
#' @param ego_init An object returned from \code{\link{Initialize}} that either includes a user-given design or a generated space-filling design as well as their corresponding function values.
#' @param x_describe An output from \code{DescribeX} that specifies the names and ranges of the variables to be optimized over. See \code{DescribeX} in \code{GaSP}.
#' @param nsteps An integer that specifies the maximum number of iterations. At every iteration, \code{EGO} queries a new design point. \code{EGO} may stop earlier than \code{nsteps} if the \code{rel_tol} in \code{control} is met for \code{wait_iter} iterations.
#' @param control A list of control parameters. See \code{\link{EGO.control}} for details.
#'
#' @return A list with the following components:
#' \item{x}{A dataframe of the values of  \eqn{\mathbf{x}} that include the initial design provided by\code{ego_init} and the points sampled by EGO.}
#' \item{y}{A vector of \eqn{f(\mathbf{x})} evaluated at \code{x} that can contain noise in non-determinstic cases.}
#' \item{n_design_init}{The number of points in the initial design.}
#' \item{val_track}{A vector of tracked values, one element for each EGO iteration. The type of values depends on the acquisition function. The minimum is obtained over all data points queried so far that include points from the initial design.
#'
#' For EI, \code{val_track} is the minimum of the function values.
#'
#' For EQI, \code{val_track} is the minimum of the estimated q-th quantile of the function values; each element is obtained from the GaSP fit at that iteration.
#'
#' For AEI, \code{val_track} is the estimated expectation of the function values evaluated at the \eqn{\mathbf{x}} that achieves the lowest estimated q-th quantile of \eqn{f(\mathbf{x})}; each element is obtained from the GaSP fit at that iteration.
#'
#' For AKG, \code{val_track} is the minimum of the estimated expectation of the function values;  each element is obtained from the GaSP fit at that iteration.}
#'
#' \item{opt_x}{The best \eqn{\mathbf{x}} found within all the queried points.}
#' \item{opt_y}{A numeric value of the function evaluated at \code{opt_x}; only provided for deterministic \code{fun}, which is when \code{type = "EI"} in  \code{acq_control} of \code{control}.}
#' \item{last_GaSP}{The object returned by the GaSP fit at the last iteration.}
#'
#' @examples
#' \dontrun{
#' rosenbrock <- function(xx) { #min is 0 at all x = 1
##'    d <- length(xx)
##'    xi <- xx[1:(d-1)]
##'    xnext <- xx[2:d]
##'    sum <- sum(100*(xnext-xi^2)^2 + (xi-1)^2)
##'    y <- sum
##'    return(y)
##' }
##'
##' fun <- rosenbrock
##' d <- 2
##' x_names <- paste("x",1:d, sep = "")
##' x_support <- rep("Continuous", d)
##' x_levels <- rep(0, d)
##' x_min <- rep(-5, d); x_max <- rep(10, d)
##' x_describe<- DescribeX(x_names, x_min, x_max, x_support, x_levels)
##' ego_init <- Initialize(x_design = NULL, y_design = NULL, n_design = 20, x_describe = x_describe, 
##' fun = fun)
##'
##' res1 <- EGO(fun, reg_model = ~1, ego_init, x_describe, nsteps = 50, control = 
##' EGO.control(alg = "genoud", rel_tol = 0, genoud_control = list(pop_size = 50, 
##' max_generations = 10)))
##'
##' res2 <- EGO(fun, reg_model = ~1, ego_init, x_describe,  nsteps = 50, control =
##'  EGO.control(alg = "direct", rel_tol = 0, direct_control = list(max_eval = 3000, finetune = TRUE)))
##' # show on the support of X
##' EGO.plot(res1, fun, n.grid = 20, x_describe = x_describe)
##'
##' # only show a section
##' EGO.plot(res1, fun, n.grid = 20, control = list(limit_min = c(-1,-1),limit_max = c(1,1)))
##'
##' # show on the support of X
##' EGO.plot(res2, fun, n.grid = 20, x_describe = x_describe)
##'
##' # only show a section
##' EGO.plot(res2, fun, n.grid = 20, control = list(limit_min = c(-1,-1),limit_max = c(1,1)))
##'
##'
##' log.goldpr  <- function(xx){  #min is log(3) at all x = (0,-1)
##'   x1 <- xx[1]
##'   x2 <- xx[2]
##'
##'   fact1a <- (x1 + x2 + 1)^2
##'   fact1b <- 19 - 14*x1 + 3*x1^2 - 14*x2 + 6*x1*x2 + 3*x2^2
##'   fact1 <- 1 + fact1a*fact1b
##'
##'   fact2a <- (2*x1 - 3*x2)^2
##'   fact2b <- 18 - 32*x1 + 12*x1^2 + 48*x2 - 36*x1*x2 + 27*x2^2
##'   fact2 <- 30 + fact2a*fact2b
##'
##'   y <- fact1*fact2
##'   return(log(y))
##' }
##' # with added noise
##' tau <- 0.1
##' fun_noise <-function(x){log.goldpr(x) + rnorm(length(log.goldpr(x)), mean = 0, sd = tau)}
##'
##' d <- 2
##' x_names <- paste("x",1:d, sep = "")
##' x_support <- rep("Continuous", d)
##' x_levels <- rep(0, d)
##' x_min <- rep(-2, d); x_max <- rep(2, d)
##' x_describe <- DescribeX(x_names, x_min, x_max, x_support, x_levels)
##' ego_init <- Initialize(x_design = NULL, y_design = NULL, n_design = 10, x_describe = x_describe, 
##' fun = fun_noise, n_rep = 10)
##' res3 <- EGO(fun_noise, reg_model = ~1, ego_init, x_describe,  nsteps = 50, control = 
##' EGO.control(alg = "genoud", rel_tol = 0, acq_control = list(type = "AEI", q= 0.9),
##' genoud_control = list(pop_size = 50, max_generations = 10)))
##' Diagnostics.plot(res3)
##' log.goldpr(res3$opt_x) # close to the true global optimum
##' res3$last_GaSP$error_var
##'
##'}
##'
#' @seealso  \code{\link{EGO.control}}, \code{\link{Initialize}}, and \code{\link{EGO.plot}}
#'
#' @export
EGO <- function(fun, reg_model, ego_init, x_describe, nsteps, control = EGO.control()){

  if(exists(".Random.seed", envir = .GlobalEnv, inherits = FALSE)){
    oldseed <- get(".Random.seed", envir = .GlobalEnv, inherits = FALSE)
    on.exit(assign(".Random.seed", oldseed, envir = .GlobalEnv))
  }

  x <- ego_init$x_design
  y <- ego_init$y_design
  GaSP_control <- control$GaSP_control
  acq_control <- control$acq_control
  val_track <- ac_val_track <- NULL  #track the value found so far: fmin, lowest quantile, lowest expectation, or lowest expectation at the point with the lowest quantile depending on the noise type
  ego_init_range <- max(ego_init$y_design) - min(ego_init$y_design)

  check.errors(reg_model, ego_init, x_describe, fun, nsteps, control)


  # missing values
  if(!("cor_family" %in% names(GaSP_control))){
    GaSP_control$cor_family <- "PowerExponential"
  }
  if(!("alpha_min" %in% names(GaSP_control))){
    GaSP_control$alpha_min <- 0.0
  }
  if(!("alpha_max" %in% names(GaSP_control))){
    GaSP_control$alpha_max <- 1.99
  }
  if(!("nugget" %in% names(GaSP_control))){
    GaSP_control$ nugget <- 1e-09
  }
  if(!("theta_standardized_min" %in% names(GaSP_control))){
    GaSP_control$theta_standardized_min <- 0
  }

  if(!("theta_standardized_max" %in% names(GaSP_control))){
    GaSP_control$theta_standardized_max <- .Machine$double.xmax  }

  if(!("derivatives_min" %in% names(GaSP_control))){
    GaSP_control$derivatives_min <- 0
  }

  if(!("derivatives_max" %in% names(GaSP_control))){
    GaSP_control$derivatives_max <- 3
  }


  if(control$alg == "direct"){
    if(!("finetune" %in% names(control$direct_control))){
      control$direct_control$finetune <- FALSE
    }

    if(!("max_eval" %in% names(control$direct_control))){
      control$direct_control$max_eval <- 3000
    }

    if(!("direct_tol" %in% names(control$direct_control))){
      control$direct_control$direct_tol <- 0
      control$direct.control$ftol_abs <- 0
    }else{
      control$direct.control$ftol_abs <-   control$direct_control$direct_tol* ego_init_range
    }
  }

  for(i in 1:nsteps){

    if(i == 1){
      cor_par <-  data.frame(0)
    }else{
      cor_par <- GaSP_fit$cor_par
    }

    sink("NULL")

    if(acq_control$type == "EI"){

       GaSP_fit <- Fit(reg_model = reg_model, x = x, y = y,   cor_par =  cor_par, cor_family = GaSP_control$cor_family,
                    random_error = FALSE, tries = 2, fit_objective = "Likelihood", nugget =  GaSP_control$nugget,
                    alpha_min = GaSP_control$alpha_min, alpha_max = GaSP_control$alpha_max,
                    theta_standardized_min = GaSP_control$theta_standardized_min,
                    theta_standardized_max = GaSP_control$theta_standardized_max,
                    derivatives_min = GaSP_control$derivatives_min,
                    derivatives_max = GaSP_control$derivatives_max,
                    model_comparison = "Objective")
    }else{
       GaSP_fit <- Fit(reg_model = reg_model, x = x, y = y,   cor_par =  cor_par, cor_family = GaSP_control$cor_family,
                       random_error = TRUE, tries = 2, fit_objective = "Likelihood", nugget =  GaSP_control$nugget,
                       alpha_min = GaSP_control$alpha_min, alpha_max = GaSP_control$alpha_max,
                       theta_standardized_min = GaSP_control$theta_standardized_min,
                       theta_standardized_max = GaSP_control$theta_standardized_max,
                       derivatives_min = GaSP_control$derivatives_min,
                       derivatives_max = GaSP_control$derivatives_max,
                       model_comparison = "Objective")
    }

    sink()

    GaSP_fit$formula <- reg_model
    comp_obj <- compute.decomposition(GaSP_fit)

    if(acq_control$type == "EI"){
      val <- min(y)
    }

    if(acq_control$type == "AKG"){
      tmp <- compute.pred(GaSP_fit, xnew = GaSP_fit$x, comp_obj = comp_obj, get_se = FALSE)
      val <- min(tmp$pred_y)
    }

    if(acq_control$type == "EQI"){
      tmp <- compute.pred(GaSP_fit, xnew = GaSP_fit$x, comp_obj = comp_obj, get_se = TRUE)
      val <- min(tmp$pred_y + qnorm(acq_control$q)*tmp$pred_se)
    }

    if(acq_control$type == "AEI"){
      tmp <- compute.pred(GaSP_fit, xnew = GaSP_fit$x, comp_obj = comp_obj, get_se = TRUE)
      val <- tmp$pred_y[which.min(tmp$pred_y + qnorm(acq_control$q)*tmp$pred_se)]
    }

    val_track <- c(val_track, val)

    if(acq_control$type == "EI"){  # no noise; use EI

      EI_fun <-  function(x){
        x <- data.frame(matrix(x, nrow = 1))
        colnames(x) <- x_describe$Variable
        EI.compute(x, val, GaSP_fit,comp_obj)
      }
      ac_fun <- EI_fun
    }

    if(acq_control$type == "AEI"){

      AEI_fun <- function(x){
        x <- data.frame(matrix(x, nrow = 1))
        colnames(x) <- x_describe$Variable
        return(AEI.compute(x, T_val = val,  GaSP_fit, comp_obj, bb = acq_control$q))     #val has been computed earlier
      }
      ac_fun <- AEI_fun
    }

    if(acq_control$type == "EQI"){

      EQI_fun <- function(x){
        x <- data.frame(matrix(x, nrow = 1))
        colnames(x) <- x_describe$Variable
        return(EQI.compute(x, q.min = val, t = nsteps +1 - i, GaSP_fit, comp_obj, bb = acq_control$q))
      }
      ac_fun <- EQI_fun
    }

    if(acq_control$type == "AKG"){

      pred_y <- compute.pred(GaSP_fit, xnew = GaSP_fit$x, comp_obj = comp_obj, get_se = FALSE)$pred_y

      r_1_tilde <- forwardsolve(comp_obj$U,  compute.r(model_fit = GaSP_fit, GaSP_fit$x, GaSP_fit$x))
      design_1 <- as.matrix(get.design.matrix(GaSP_fit, GaSP_fit$x))
      f_1_tilde <- forwardsolve(t(comp_obj$r_tilde), t(design_1))

      AKG_fun <- function(x){
        x <- data.frame(matrix(x, nrow = 1))
        colnames(x) <- x_describe$Variable
        return(AKG.compute(x, pred_y, GaSP_fit, r_1_tilde, f_1_tilde, comp_obj))
      }
      ac_fun <- AKG_fun
    }


    tt <- system.time(x_next <- ac.findmax(ac_fun, x_describe, control))

    ac_val_track <- c(ac_val_track, ac_fun(x_next))
    if(check.stop(control, ego_init_range, ac_val_track) == TRUE){   # check if met the stopping criterion
       break
    }

    #take a new sample
    x <- rbind(x, x_next)
    y <- c(y, fun(x_next))

    if(control$print_level){
      print(paste("The", i,"th iteration val =", val, sep = " "))
    }
  }

  sink("NULL")
  if(acq_control$type == "EI"){
    last_GaSP <-  Fit(reg_model = reg_model, x = x, y = y, cor_family = GaSP_control$cor_family,
                                random_error = FALSE, fit_objective = "Likelihood", nugget =  GaSP_control$nugget,
                                alpha_min = GaSP_control$alpha_min, tries = 2, alpha_max = GaSP_control$alpha_max, model_comparison = "Objective")
  }else{
    last_GaSP <-  Fit(reg_model = reg_model, x = x, y = y, cor_family = GaSP_control$cor_family,
                      random_error = TRUE, fit_objective = "Likelihood", nugget =  GaSP_control$nugget,
                      alpha_min = GaSP_control$alpha_min, tries = 2, alpha_max = GaSP_control$alpha_max, model_comparison = "Objective")
  }
  sink()

  last_GaSP$formula <- reg_model
  last_comp_obj <- compute.decomposition(last_GaSP)

  if(acq_control$type == "EI"){
    opt_x <-  x[which.min(y),]
    opt_y <-  min(y)
  }

  if(acq_control$type == "AKG"){
    tmp <- compute.pred(last_GaSP, xnew = last_GaSP$x, comp_obj = last_comp_obj, get_se = FALSE)
    opt_x <-last_GaSP$x[which.min(tmp$pred_y),]
  }

  if(acq_control$type == "EQI"){
    tmp <- compute.pred(last_GaSP, xnew = last_GaSP$x, comp_obj = last_comp_obj, get_se = TRUE)
    opt_x <-last_GaSP$x[which.min(tmp$pred_y + qnorm(acq_control$q)*tmp$pred_se),]
  }

  if(acq_control$type == "AEI"){
    tmp <- compute.pred(last_GaSP, xnew = last_GaSP$x, comp_obj = last_comp_obj, get_se = TRUE)
    opt_x <- last_GaSP$x[which.min(tmp$pred_y + qnorm(acq_control$q)*tmp$pred_se),]

  }

  dat_obj <- list(x = x, y = y,  n_design_init = nrow(ego_init$x_design),
                  val_track = val_track,
                  opt_x = opt_x, last_GaSP =  last_GaSP)

  if(acq_control$type == "EI"){
    dat_obj  <- c(dat_obj, list(opt_y = opt_y))
  }
  return(dat_obj)
}


compute_cor_matern <- function(derivatives, theta, distance){

  if(derivatives == 0){
    return(-theta*distance)
  }
  if(derivatives == 1){
    tmp <- theta*distance
    return(-theta*distance + log(theta*distance +1))
  }
  if(derivatives == 2){
    return(-theta*distance + 2*log(theta*distance +1))

  }
  if(derivatives == 3){
    return(-theta*distance^2)
  }
}

compute.r <- function(model_fit, x, xnew){

  cor_par <- model_fit$cor_par

  if(model_fit$cor_family=="PowerExponential"){

    R_log <- matrix(0, nrow(x), nrow(xnew))

    for(i in 1:ncol(x)){

     distance <- abs(outer(x[,i],xnew[,i],'-'))
     tmp <-  -cor_par[i,1]*(distance^{2-cor_par[i,2]})
     R_log <- R_log+tmp
   }
    R <- exp(R_log)
  }

  if(model_fit$cor_family == "Matern"){

    R_log <- matrix(0, nrow(x), nrow(xnew))

    for(i in 1:ncol(x)){
      distance <- abs(outer(x[,i],xnew[,i],'-'))
      tmp <- compute_cor_matern(cor_par$Derivatives[i], cor_par$Theta[i], distance)
      R_log <- R_log+tmp
    }
    R <- exp(R_log)
  }

  return(R)
}




# create the design matrix
get.design.matrix <- function(model_fit, x){

  return(model.matrix(model_fit$formula, data = x))
}

compute.decomposition <- function(model_fit){

    R <- compute.r(model_fit, model_fit$x, model_fit$x)
    design_train <- as.matrix(get.design.matrix(model_fit, model_fit$x))
    #res <- model_fit$y - design_train%*%as.matrix(model_fit$beta)
    R_tilde <- R + diag(model_fit$error_var/(model_fit$error_var + model_fit$sp_var),nrow(R))
    U <- t(chol(R_tilde))
    f_tilde <- forwardsolve(U, design_train)
    y_tilde <- forwardsolve(U, model_fit$y)
    qr_obj <- qr(f_tilde)
    q_tilde <- qr.Q(qr_obj)
    r_tilde <- qr.R(qr_obj)
    c_vec <- backsolve(t(U), y_tilde - q_tilde%*%t(q_tilde)%*%y_tilde)
    return(list(c_vec = c_vec, U = U, r_tilde = r_tilde, q_tilde = q_tilde))
}

AKG.compute <- function(x_new, pred_y,  GaSP_fit, r_1_tilde, f_1_tilde,  comp_obj){

  pred_tmp <- compute.pred(GaSP_fit, xnew = x_new, comp_obj = comp_obj,  get_se = TRUE)
  min_term_1 <-  min(c(pred_y, pred_tmp$pred_y))

  a_i <- c(pred_y,pred_tmp$pred_y)

  #b_i <- apply(GaSP_fit$x, 1, fun_tmp)/sqrt(GaSP_fit$sp_var+GaSP_fit$error_var)
  b_i <- compute.cov.train(model_fit = GaSP_fit, x_new, r_1_tilde, f_1_tilde, comp_obj)
  b_i <- c(b_i,  pred_tmp$pred_se^2)/sqrt(pred_tmp$pred_se^2 + GaSP_fit$error_var)

  ## then sort a and b
  a_i <- -a_i
  Isort <- order(x=b_i,y=a_i)
  Iremove <- numeric()
  a <- a_i[Isort]
  b <- b_i[Isort]

  nobs <- length(a_i) - 1

  for (i in 1:(nobs)){
    if (b[i+1] == b[i]){
      Iremove <- c(Iremove, i)  #set of points with the same b
      }
  }

  if (length(Iremove) > 0) {
      b <- b[-Iremove]  #remove this point...
      a <- a[-Iremove]
   }

   # update this
    nobs <- length(a)-1

    C <- rep(0, nobs+2)
    C[1] <- -1e36
    C[length(C)] <- 1e36
    A1 <- 0

    for(k in 2:(nobs+1)){
      nondom <- 1
       if(k == nobs+1){
         nondom <- 1
       }else{
         if((a[k+1] >= a[k]) && (b[k] == b[k+1])){   # exit a dominant pair
           nondom <- 0
           }
        }
      if(nondom == 1){
        loopdone <- 0
        count <- 0
        while ( loopdone == 0 && count < 1e3 ){
          count <- count + 1
          u <- A1[length(A1)] + 1
          C[u+1] <- (a[u]-a[k]) / (b[k] - b[u])
          if ((length(A1) > 1) && (C[u+1] <= C[A1[length(A1)-1]+2])){
            A1 <- A1[-length(A1)]
           }else{
            A1 <- c(A1, k-1)
            loopdone <- 1
        }
      }
    }
  }
  at <- a[A1+1]
  bt <- b[A1+1]
  ct <- C[c(1, A1+2)]

  min_term_2  <- 0
  for (k in 1:length(at)){
    min_term_2  <-  min_term_2  + at[k]*(pnorm(ct[k+1])-pnorm(ct[k])) + bt[k]*(dnorm(ct[k]) - dnorm(ct[k+1]))
  }

  return(min_term_2 +  min_term_1)
}

#outputs a vector .....
compute.cov.train <- function(model_fit, x_new, r_1_tilde, f_1_tilde, comp_obj = NULL){

  term1 <- compute.r(model_fit, model_fit$x, x_new)

  r_2_tilde <- forwardsolve(comp_obj$U,  compute.r(model_fit, model_fit$x, x_new))
  term2 <-  -t(r_1_tilde)%*% r_2_tilde

  design_2 <- as.matrix(get.design.matrix(model_fit, x_new))
  f_2_tilde <- forwardsolve(t(comp_obj$r_tilde), t(design_2))

  term3 <- (t(f_1_tilde -t(comp_obj$q_tilde)%*%r_1_tilde)%*%(f_2_tilde -t(comp_obj$q_tilde)%*%r_2_tilde))

  return((model_fit$error_var + model_fit$sp_var)*(term1 + term2 + term3))
}


# actually don't need this function .....
compute.cov <- function(model_fit,  x_1, x_2, comp_obj = NULL){

  term1 <- compute.r(model_fit, x_1, x_2)

  r_1_tilde <- forwardsolve(comp_obj$U,  compute.r(model_fit, model_fit$x, x_1))
  r_2_tilde <- forwardsolve(comp_obj$U,  compute.r(model_fit, model_fit$x, x_2))

  term2 <-  -t(r_1_tilde)%*% r_2_tilde

  design_1 <- as.matrix(get.design.matrix(model_fit, x_1))
  design_2 <- as.matrix(get.design.matrix(model_fit, x_2))

  f_1_tilde <- forwardsolve(t(comp_obj$r_tilde), t(design_1))
  f_2_tilde <- forwardsolve(t(comp_obj$r_tilde), t(design_2))

  term3 <- ((f_1_tilde -t(comp_obj$q_tilde)%*%r_1_tilde)*(f_2_tilde -t(comp_obj$q_tilde)%*%r_2_tilde))

  return((model_fit$error_var + model_fit$sp_var)*(term1 + term2 + term3))
}


compute.pred <- function(model_fit,  xnew, comp_obj, get_se = TRUE){

  rnew <- compute.r(model_fit,model_fit$x, xnew)
  design_new <- as.matrix(get.design.matrix(model_fit, xnew))

  pred_y <-   as.numeric(design_new %*%as.matrix(model_fit$beta) + t(rnew)%*%comp_obj$c_vec)

  if(get_se){
    f_te_tilde <- forwardsolve(t(comp_obj$r_tilde), t(design_new))
    rnew_tilde <- forwardsolve(comp_obj$U, rnew)
    pred_se <- sqrt((model_fit$error_var + model_fit$sp_var)*(1- apply(rnew_tilde, 2, crossprod) +  apply(f_te_tilde -t(comp_obj$q_tilde)%*%rnew_tilde, 2, crossprod)))

    return(list(pred_y = pred_y, pred_se = pred_se))
  }else{
    return(list(pred_y = pred_y))
  }
}


check.stop <- function(control, ego_init_range, ac_val_track){

  if(length(ac_val_track)> control$wait_iter){
    return(sum(tail(ac_val_track,control$wait_iter) < control$rel_tol*ego_init_range) ==control$wait_iter)
  }
  return(FALSE)
}

EI.compute <- function(x_new, fmin, GaSP_fit, comp_obj){

    pred_tmp <- compute.pred(GaSP_fit, xnew = x_new, comp_obj = comp_obj, get_se = TRUE)
    return((fmin - pred_tmp$pred_y)*pnorm((fmin - pred_tmp$pred_y)/pred_tmp$pred_se) +
             pred_tmp$pred_se*dnorm((fmin - pred_tmp$pred_y)/pred_tmp$pred_se))

}

AEI.compute <- function(x_new, T_val, GaSP_fit, comp_obj, bb = 0.75){

  pred_tmp <- compute.pred(GaSP_fit, xnew = x_new, comp_obj = comp_obj, get_se = TRUE)
  EI_val <- (T_val - pred_tmp$pred_y)*pnorm((T_val - pred_tmp$pred_y)/pred_tmp$pred_se) +
    pred_tmp$pred_se*dnorm((T_val - pred_tmp$pred_y)/pred_tmp$pred_se)
  tau_est <- sqrt(GaSP_fit$error_var)
  return( EI_val*(1 - tau_est/sqrt(pred_tmp$pred_se^2 + tau_est^2)))
}


EQI.compute <- function(x_new, q.min,  t,  GaSP_fit, comp_obj, bb = 0.75){

  pred_tmp <- compute.pred(GaSP_fit, xnew = x_new, comp_obj = comp_obj,  get_se = TRUE)
  tau_new <- sqrt(GaSP_fit$error_var) /sqrt(t)
  m_q <-  pred_tmp$pred_y + qnorm(bb)* sqrt((tau_new^2 * pred_tmp$pred_se^2)/(tau_new^2 + pred_tmp$pred_se^2))
  tmp <- q.min -  m_q
  S_q <- pred_tmp$pred_se^2/sqrt(tau_new^2 + pred_tmp$pred_se^2)

  return( tmp*pnorm(tmp/S_q) + S_q* dnorm(tmp/S_q))
}


ac.findmax <- function(ac_fun, x_describe, control){


  if(control$alg == "direct"){


    opts <- list("maxeval" = control$direct_control$max_eval, "print_level"  = 0, "algorithm"="NLOPT_GN_DIRECT_L","xtol_rel"=0, "xtol_abs" = 0, ftol_abs=control$direct.control$ftol_abs)

    # not useful for direct
    init_point <- transform.design(randomLHS(1,nrow(x_describe)), x_describe)
    myTryCatch_output <- myTryCatch(optim_obj  <- nloptr(x0=init_point,eval_f=function(x){-ac_fun(x)},lb =x_describe$Min,ub = x_describe$Max, opts=opts))
    optim_obj$par <-  optim_obj$solution
    optim_obj$value <- optim_obj$objective

    if(control$print_level == 2){
      if(!is.null(myTryCatch_output$warning))
        print(myTryCatch_output$warning)
    }


    if(control$direct_control$finetune){

      optim_tune_obj <- tryCatch(optim(par = optim_obj$par, fn = function(x){-ac_fun(x)},lower =x_describe$Min,
                                  upper = x_describe$Max, method = "L-BFGS-B", control = control$optim_control),  error = function(x){return(1)})

      if(length(optim_tune_obj)!=1){
        if((-optim_tune_obj$value)>=(-optim_obj$value)){
          return(optim_tune_obj$par)
        }else{
        }
      }
      return(optim_obj$par)
      }
    return(optim_obj$par)
  }

  if(control$alg == "genoud"){

    d <-  length(x_describe$Variable)
    domaine <- cbind(x_describe$Min,  x_describe$Max)
    parinit <- x_describe$Min + runif(d)*(x_describe$Max-x_describe$Min)

    pop.size = 50; max.generations = 12; wait.generations = 2; BFGSburnin = 2; solution.tolerance=1e-21; BFGSmaxit = pop.size
    P1=50; P2=50; P3=50; P4=50; P5=50; P6=50; P7=50; P8=50; P9=0; P9mix=NULL; gr = NULL


    list2env(control$genoud_control, envir = environment())

    if("max_generations" %in% names(control$genoud_control)){
      max.generations = control$genoud_control$max_generations
    }
    if("pop_size" %in% names(control$genoud_control)){
      pop.size = control$genoud_control$pop_size
    }

    myTryCatch_output <- myTryCatch(genoud_obj <- genoud(fn = function(x){ac_fun(x)}, nvars=d, max=TRUE,
                         pop.size=pop.size, max.generations=max.generations, wait.generations=wait.generations,
                         hard.generation.limit=FALSE, starting.values=parinit, MemoryMatrix=TRUE,
                         Domains=domaine,  solution.tolerance=solution.tolerance,
                         gr = NULL, boundary.enforcement=2, lexical=FALSE, gradient.check=FALSE, BFGS=TRUE,
                         data.type.int=FALSE, hessian=FALSE, unif.seed=floor(runif(1,max=10000)), int.seed=floor(runif(1,max=10000)),
                         print.level=0, share.type=0, instance.number=0, output.path="stdout", output.append=FALSE, project.path=NULL,
                         P1=P1, P2=P2, P3=P3, P4=P4, P5=P5, P6=P6, P7=P7, P8=P8, P9=P9, P9mix=P9mix,
                         BFGSburnin=BFGSburnin, BFGSfn=NULL, BFGShelp=NULL, control=control$optim_control,
                         cluster=FALSE, balance=FALSE, debug=FALSE))

    if(control$print_level == 2){
      if(!is.null(myTryCatch_output$warning))
        print(myTryCatch_output$warning)
    }
    return(genoud_obj$par)
  }
}


# transform unit hypercube LHS design to match with x_describe
transform.design <- function(lhs_design, x_describe){
  x_design <- scaled_design <- mapply(function(i){lhs_design[,i]*(x_describe$Max[i] - x_describe$Min[i])+x_describe$Min[i] }, 1:ncol(lhs_design))
  idx <- which(x_describe$Support == "Grid")
  if(length(idx)>0){
    for(i in idx){
      rr <- seq(x_describe$Min[i],x_describe$Max[i], length.out = x_describe$NumberLevels[i])
      midpoints <- head(rr,-1) + diff(rr)/2         # find midpoints between reference points
      intv <- findInterval( scaled_design[,i], midpoints)
      x_design[,i] <- rr[intv+1]
    }
  }
  return(x_design)
}



do.silent=function(code){
  sink("NUL") # use /dev/null in UNIX
  tmp = code
  sink()
  return(tmp)
}
