
#' Visualize EGO results in 2D 
#' 
#' A function to plot the countour, initial design, and sampled design points for optimizing 2D functions. 
#' 
#' 
#' Green points corresponds to the initial design and blue points are the points queried by EGO.  In addition to the control parameters, the user can also specify to overwrite \code{nlevels} the default of \code{filled.contour}. 
#' 
#'@param ego_fit An object returned from \code{EGO}
#'@param fun The function \code{ego_fit} was fit to minimize. \code{fun} needs to be a deterministic function. 
#'@param n.grid The number of grid points in each direction to evaluate  \code{fun} and make the countour plot. 
#'@param x_describe  An output from \code{DescribeX} that specifies the names and ranges of the variables to be optimized over. Not required if \code{limit_min} and \code{limit_max} are provided in \code{control}. 
#'@param control A list of control parameters  
#'@export
EGO.plot <- function(ego_fit, fun, n.grid = 20, x_describe=NULL, control = list(limit_min = NULL, limit_max = NULL, label_order = FALSE)){
  
  if(missing(control)){
    x1.lim.min<- x_describe$Min[1]
    x2.lim.min<- x_describe$Min[2]
  } else{
    x1.lim.min<- control$limit_min[1]
    x2.lim.min<- control$limit_min[2]
  }
  if(missing(control)){
    x1.lim.max<- x_describe$Max[1]
    x2.lim.max<- x_describe$Max[2]
  }else{
    x1.lim.max<- control$limit_max[1]
    x2.lim.max<-control$limit_max[1]
  }
  
  x1.grid <-  seq( x1.lim.min, x1.lim.max,length=n.grid)
  x2.grid <- seq(x2.lim.min, x2.lim.max,length=n.grid) 
  design.grid <- expand.grid(x1.grid, x2.grid)
  response.grid <-  matrix(apply(design.grid, 1,  fun), n.grid, n.grid)
  
  ## parameters set to default values 
  zlim = range(response.grid, finite = TRUE)
  nlevels = 20
  levels = pretty(zlim, nlevels)
  color.palette = function(n) hcl.colors(n, "YlOrRd", rev = TRUE)
  col = color.palette(length(levels) - 1)
  
  list2env(control, envir = environment())
  
  filled.contour(x = x1.grid, y = x2.grid, z =  response.grid,  color.palette  =  color.palette,
                 col = col,    nlevels =  nlevels,      levels  =   levels ,       plot.axes = {axis(1);
    axis(2);contour(x1.grid, x2.grid, response.grid, add = TRUE); points(ego_fit$x[1:ego_fit$n_design_init,1], ego_fit$x[1:ego_fit$n_design_init,2],cex = 1.5,  pch=17, col="green"); 
    points(ego_fit$x[(ego_fit$n_design_init+1):nrow(ego_fit$x),1], ego_fit$x[(ego_fit$n_design_init+1):nrow(ego_fit$x),2],cex = 1.5,  pch=17, col="blue") 
  })
  
}

