#' A function to initialize EGO
#' 
#' This function generates the \code{design} input of \code{EGO}. It either specifies an user given design or generate a Latin Hypercube space-filling design given the objective function and the number of initial points. 
#' 
#' @param x_design A data frame with row corresponding to one design points; required if the user provides a design; not required if the user uses an automatically generated design. 
#' @param y_design  A vector of function values evaluated at \code{x_design}; optional if \code{fun} is provided. 
#' @param n_design  The number of points in the initial design; required if the user uses a generated design (when \code{x_design} is not provided). 
#' @param x_describe  An output from \code{DescribeX} that specifies the names and ranges of the variables to be optimized over. See \code{DescribeX} in \code{GaSP}. 
#' @param fun A function to be minimized. It can be deterministic or with added random noise.
#' @param n_rep The number of replicates in the initial design. \code{rep} points are randomly sampled with replacement and evaluated again to add to the initial design. The total number of design points is \code{n_design} + \code{rep}. This option is useful for noisy settings since the replicates help to estimate the variance of random errors. 
#' @return A list with the following components:
#' \item{x_design}{A data frame of the initial design; either user provided or generated by \code{lhs}.}
#' \item{y_design}{A vector of function values evaluated at \code{x_design}.} 
#' @export
Initialize <- function(x_design = NULL, y_design = NULL, n_design, x_describe = NULL, fun = NULL, n_rep = 0){
  
  # x_design provided but not y_design
  if((!is.null(x_design)) & is.null(y_design)){
    y_design <- apply(x_design, 1, fun)
  }
  
  if(is.null(x_design) & is.null(y_design)){  # generate a design with 10*d intial poitns
    lhs_design <-randomLHS(n_design,nrow(x_describe))
    x_design <- transform.design(lhs_design, x_describe)
    if(n_rep > 0){
      x_design <- rbind(x_design, x_design[sample(1:nrow(x_design), n_rep),])
    }
    y_design <- apply(x_design, 1, fun)
  }
  
  if(is.null(x_design) | is.null(y_design)){
    stop("both x_design and y_design cannot be NULL!")
  }
  
  colnames(x_design) = x_describe$Variable
  x_design <- data.frame(x_design)
  return(list(x_design = x_design, y_design  = as.vector(y_design)))
}
