#' Diagnostic plots of the EGO fits
#' 
#' A function to make diagnostic plots of EGO based on the last GaSP fit.

#' 
#' This function generates four figures:
#'
#'  (a) the predicted function value evaluated at the sampled points versus the sampled function values
#'  
#'  (b) the predicted function value evaluated at the sampled points versus the residuals 
#'  
#'  (c) the predicted function value evaluated at the sampled points versus the standardized residuals 
#'  
#'  (d) the QQ-plot of the standardized residuals (the standard normal quantiles versus the standardized residuals)
#'  

#'@param ego_fit An object returned from \code{EGO}
#'@param cv An logical value indicating if the predicted values are computed using leave-one-out cross validation or not. For the latter,  predictions are made using GaSP fitted with all data avaiable at the last iteration before the termination of EGO.  

#' @export
#' 
Diagnostics.plot <- function(ego_fit, cv = TRUE){
  #predict versus y
  par(mfrow = c(2,2))
  last_y <-  ego_fit$y
  
  if(!cv){
    tmp <- suppressWarnings(Predict(x_pred = ego_fit$x, ego_fit$last_GaSP, generate_coefficients = FALSE)$y_pred)
    plot(last_y~tmp[,1],  ylab = "y", xlab = "pred y")
    abline(a = 0, b= 1, col = "red")
    plot(tmp[,1] - last_y ~tmp[,1], xlab = "pred y", ylab = "residuals")
    abline(a = 0, b= 0, col = "red")
    plot(y = (last_y-tmp[,1])/tmp[,2], x = tmp[,1], xlab = "pred y", ylab = "standardized residuals")
    abline(a = 0, b= 0, col = "red")
    qqnorm( (last_y-tmp[,1])/tmp[,2],xlab = "standard normal quantiles",
            ylab =  "standardized residuals",  main = ' ')
    qqline((last_y-tmp[,1])/tmp[,2], col = "red", lwd = 2)
    
  }else{
    GaSP_cv <- suppressWarnings(CrossValidate(ego_fit$last_GaSP))
    
    plot(y = last_y, x = GaSP_cv[,1],   ylab = "y", xlab = "pred y (cv)")
    abline(a = 0, b= 1, col = "red")
    
    plot(GaSP_cv[,1] - last_y ~GaSP_cv[,1] ,    xlab = "pred y (cv)", ylab = "residuals (cv)")
    abline(a = 0, b= 0, col = "red")
    
    plot(y = (last_y-GaSP_cv[,1])/GaSP_cv[,2], x = GaSP_cv[,1], xlab = "pred y (cv)", ylab = "standardized residuals (cv)")
    abline(a = 0, b= 0, col = "red")
    
    qqnorm( (last_y-GaSP_cv[,1])/GaSP_cv[,2],xlab = "standard normal quantiles",
            ylab =  "standardized residuals (cv)", main = ' ')
    qqline((GaSP_cv[,1] - last_y)/GaSP_cv[,2], col = "red", lwd = 2 )
  }
}

