% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/EGO.R
\name{EGO}
\alias{EGO}
\title{Efficient Global Optimization (EGO)}
\usage{
EGO(fun, reg_model, ego_init, x_describe, nsteps, control = EGO.control())
}
\arguments{
\item{fun}{A function to be minimized. It can be deterministic or with added random noise.}

\item{reg_model}{A formula object that specifies the regression model passed to the \code{Fit} function in the \code{GaSP} package.}

\item{ego_init}{An object returned from \code{\link{Initialize}} that either includes a user-given design or a generated space-filling design as well as their corresponding function values.}

\item{x_describe}{An output from \code{DescribeX} that specifies the names and ranges of the variables to be optimized over. See \code{DescribeX} in \code{GaSP}.}

\item{nsteps}{An integer that specifies the maximum number of iterations. At every iteration, \code{EGO} queries a new design point. \code{EGO} may stop earlier than \code{nsteps} if the \code{rel_tol} in \code{control} is met for \code{wait_iter} iterations.}

\item{control}{A list of control parameters. See \code{\link{EGO.control}} for details.}
}
\value{
A list with the following components:
\item{x}{A dataframe of the values of  \eqn{\mathbf{x}} that include the initial design provided by\code{ego_init} and the points sampled by EGO.}
\item{y}{A vector of \eqn{f(\mathbf{x})} evaluated at \code{x} that can contain noise in non-determinstic cases.}
\item{n_design_init}{The number of points in the initial design.}
\item{val_track}{A vector of tracked values, one element for each EGO iteration. The type of values depends on the acquisition function. The minimum is obtained over all data points queried so far that include points from the initial design.

For EI, \code{val_track} is the minimum of the function values.

For EQI, \code{val_track} is the minimum of the estimated q-th quantile of the function values; each element is obtained from the GaSP fit at that iteration.

For AEI, \code{val_track} is the estimated expectation of the function values evaluated at the \eqn{\mathbf{x}} that achieves the lowest estimated q-th quantile of \eqn{f(\mathbf{x})}; each element is obtained from the GaSP fit at that iteration.

For AKG, \code{val_track} is the minimum of the estimated expectation of the function values;  each element is obtained from the GaSP fit at that iteration.}

\item{opt_x}{The best \eqn{\mathbf{x}} found within all the queried points.}
\item{opt_y}{A numeric value of the function evaluated at \code{opt_x}; only provided for deterministic \code{fun}, which is when \code{type = "EI"} in  \code{acq_control} of \code{control}.}
\item{last_GaSP}{The object returned by the GaSP fit at the last iteration.}
}
\description{
EGO performs Bayesian optimization using the EGO algorithm.
The objective is to find the \eqn{\mathbf{x}} that minimizes \eqn{f(\mathbf{x})}.
Two cases are considered: deterministic or noisy.  In the former case,
the available function evaluation \eqn{y(\mathbf{x}) = f(\mathbf{x})} is accessible given any query point \eqn{\mathbf{x}}, and in the latter  \eqn{y(\mathbf{x}) = f(\mathbf{x}) + \epsilon} is acceesible  where \eqn{\epsilon} is some random noise.
}
\examples{
\dontrun{
rosenbrock <- function(xx) { #min is 0 at all x = 1
   d <- length(xx)
   xi <- xx[1:(d-1)]
   xnext <- xx[2:d]
   sum <- sum(100*(xnext-xi^2)^2 + (xi-1)^2)
   y <- sum
   return(y)
}

fun <- rosenbrock
d <- 2
x_names <- paste("x",1:d, sep = "")
x_support <- rep("Continuous", d)
x_levels <- rep(0, d)
x_min <- rep(-5, d); x_max <- rep(10, d)
x_describe<- DescribeX(x_names, x_min, x_max, x_support, x_levels)
ego_init <- Initialize(x_design = NULL, y_design = NULL, n_design = 20, x_describe = x_describe, 
fun = fun)

res1 <- EGO(fun, reg_model = ~1, ego_init, x_describe, nsteps = 50, control = 
EGO.control(alg = "genoud", rel_tol = 0, genoud_control = list(pop_size = 50, 
max_generations = 10)))

res2 <- EGO(fun, reg_model = ~1, ego_init, x_describe,  nsteps = 50, control =
 EGO.control(alg = "direct", rel_tol = 0, direct_control = list(max_eval = 3000, finetune = TRUE)))
# show on the support of X
EGO.plot(res1, fun, n.grid = 20, x_describe = x_describe)

# only show a section
EGO.plot(res1, fun, n.grid = 20, control = list(limit_min = c(-1,-1),limit_max = c(1,1)))

# show on the support of X
EGO.plot(res2, fun, n.grid = 20, x_describe = x_describe)

# only show a section
EGO.plot(res2, fun, n.grid = 20, control = list(limit_min = c(-1,-1),limit_max = c(1,1)))


log.goldpr  <- function(xx){  #min is log(3) at all x = (0,-1)
  x1 <- xx[1]
  x2 <- xx[2]

  fact1a <- (x1 + x2 + 1)^2
  fact1b <- 19 - 14*x1 + 3*x1^2 - 14*x2 + 6*x1*x2 + 3*x2^2
  fact1 <- 1 + fact1a*fact1b

  fact2a <- (2*x1 - 3*x2)^2
  fact2b <- 18 - 32*x1 + 12*x1^2 + 48*x2 - 36*x1*x2 + 27*x2^2
  fact2 <- 30 + fact2a*fact2b

  y <- fact1*fact2
  return(log(y))
}
# with added noise
tau <- 0.1
fun_noise <-function(x){log.goldpr(x) + rnorm(length(log.goldpr(x)), mean = 0, sd = tau)}

d <- 2
x_names <- paste("x",1:d, sep = "")
x_support <- rep("Continuous", d)
x_levels <- rep(0, d)
x_min <- rep(-2, d); x_max <- rep(2, d)
x_describe <- DescribeX(x_names, x_min, x_max, x_support, x_levels)
ego_init <- Initialize(x_design = NULL, y_design = NULL, n_design = 10, x_describe = x_describe, 
fun = fun_noise, n_rep = 10)
res3 <- EGO(fun_noise, reg_model = ~1, ego_init, x_describe,  nsteps = 50, control = 
EGO.control(alg = "genoud", rel_tol = 0, acq_control = list(type = "AEI", q= 0.9),
genoud_control = list(pop_size = 50, max_generations = 10)))
Diagnostics.plot(res3)
log.goldpr(res3$opt_x) # close to the true global optimum
res3$last_GaSP$error_var

}

}
\seealso{
\code{\link{EGO.control}}, \code{\link{Initialize}}, and \code{\link{EGO.plot}}
}
